/**********************************************************************

Filename    :   Direct3DWin32App.h
Content     :   Simple Direct3D 9 Win32 Application class
Created     :   
Authors     :   Michael Antonov
Copyright   :   (c) 2005-2006 Scaleform Corp. All Rights Reserved.

This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING 
THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR ANY PURPOSE.

**********************************************************************/

#ifndef INC_D3DWIN32APP_H
#define INC_D3DWIN32APP_H

#include <windows.h>

// GFx Includes
#include "GTypes.h"

// ***** Direct3D9 Includes

#if defined(GFC_BUILD_DEFINE_NEW) && defined(GFC_DEFINE_NEW)
#undef new
#endif

#include <d3d9.h>
#include <d3dx9.h>

#if defined(GFC_BUILD_DEFINE_NEW) && defined(GFC_DEFINE_NEW)
#define new GFC_DEFINE_NEW
#endif




// ***** Application class

class Direct3DWin32App
{
public:

    // **** Public Interface

    Direct3DWin32App();
    ~Direct3DWin32App();
    
    // Creates a window & initializes the application class.
    // Returns 0 if window creation/initialization failed (the app should quit).
    bool            SetupWindow(const char *pname, SInt width, SInt height);

    // Message processing function to be called in the 
    // application loops until it returns 0, to indicate application being closed.
    bool            ProcessMessages();

    // Sleeps for the specified number of milliseconds or till message.
    void            SleepTillMessage(UInt32 ms);

    // This call cleans up the application and kills the window.
    // If not called explicitly, it will be called from the destructor.
    void            KillWindow();

    // Changes/sets window title
    void            SetWindowTitle(const char *ptitle);

    // Presents the data (swaps the back and front buffers)
    void            PresentFrame();

    // Resets the direct3D, return 1 if successful.
    // On successful reset, this function will call InitRenderer again.
    bool            RecreateRenderer();


    // *** Overrides

    // This override is called from SetupWindow to initialize OpenGL/Direct3D view
    virtual bool    InitRenderer();
    // Should/can be called every frame to prepare the render, user function
    virtual void    PrepareRendererForFrame();
    
    // Input overrides, invoked during the ProcessMessages call.
    virtual void    OnKey(UInt key, UInt info, bool downFlag);
    virtual void    OnChar(UInt32 wcharCode, UInt info);
    virtual void    OnMouseButton(UInt button, bool downFlag, SInt x, SInt y);
    virtual void    OnMouseWheel(SInt zdelta, SInt x, SInt y);
    virtual void    OnMouseMove(SInt x, SInt y);

    // Sizing; by default, re-initalizes the renderer
    virtual void    OnSize(SInt w, SInt h);
    // Called when sizing begins and ends.
    virtual void    OnSizeEnter(bool enterSize);
    // Handle dropped file
    virtual void    OnDropFiles(char *path);

    
    // Sets the new window size and style for full screen, optionally also resetting renderer.
    // If recreating renderer, will assign Width, Height in present params and no other values.
    // Returns 1 if recreate succeeded, 0 if it failed. If reset is not specified, function will never fail.
    bool            ConfigureWindow(SInt x, SInt y, SInt w, SInt h, bool fullScreen, bool recreateRenderer = 1);

    
    // *** Simple 2D painting and output, for displaying HUD/Stat overlays

    // Clears the entire back buffer.
    void            Clear(UInt32 color);
    // Initialize and restore 2D rendering view.
    // Push2DRenderView must be called before using 2D functions below,
    // while Pop2DRenderView MUST be called after it is finished.
    void            Push2DRenderView();
    void            Pop2DRenderView();
    // Draw a filled + blended rectangle.
    void            FillRect(SInt l, SInt t, SInt r, SInt b, UInt32 color);
    // Draw a text string (specify top-left corner of characters, NOT baseline)
    void            DrawText(SInt x, SInt y, const char *ptext, UInt32 color);
    // Compute text size that will be generated by DrawText
    void            CalcDrawTextSize(SInt *pw, SInt *ph, const char *ptext);

    // API-independent toggle for wireframe rendering.
    void            SetWireframe(bool wireframeEnable);

    // Set cursor
    void            SetCursor(HCURSOR cursor);
    void            ShowCursor(bool show = true);
    void            ShowCursorInstantly(bool show);
    void            ResetCursor();

    // *** Implementation


    // Boolean states for the window and app
    bool            Created;
    bool            Active; 
    bool            QuitFlag;
    bool            LockOnSize;
    SInt            ExitCode;
    // Cursor states
    bool            CursorHidden;
    bool            CursorHiddenSaved;
    bool            CursorDisabled;
    bool            CursorIsOutOfClientArea;
    // Set this if the window is to accept drag and drop
    bool            SupportDropFiles;
    bool            SizableWindow;

    // Requested 3D state
    bool            FullScreen;
    bool            FSAntialias;
    UInt            BitDepth;
    // This flag will be set after SetupWindow to indicate that FSAA is available.
    bool            FSAASupported;

    // Handles..    
    HWND            hWND;
    HINSTANCE       hInstance;
    HCURSOR         hCursor;

    LPDIRECT3D9             pD3D;
    IDirect3DDevice9*       pDevice;
    D3DPRESENT_PARAMETERS   PresentParams;
    // Client size
    int                     Width;
    int                     Height;

    // Text/ 2D view rendering support functionality
    bool                    In2DView;
    bool                    WasInScene;
    ID3DXFont*              pTextFont;
    ID3DXSprite*            pTextFontSprite;
    D3DVIEWPORT9            ViewportSave;
    
    // Member window procedure, used so that it can access locals
    virtual     LRESULT     MemberWndProc(UINT message, WPARAM wParam, LPARAM lParam);

};


#endif // INC_D3DWIN32APP_H